/************************************************************************
 * @file: AudioFader.h
 *
 * @description: This header file contains definition for class Fader.
 * Fader is a Post Processing utility class for applying FADE_IN/FADE_OUT on
 * PCM data. Fader class will be used by Backend libraries for applying Fading
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2015
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOFADER_H_
#define _ADIT_UTILITY_AUDIOFADER_H_

#include <stdint.h>
#include <ostream>

#include "Logging.h"
#include "AudioTypes.h"


#define FALSE 0
#define TRUE 1

namespace adit
{

namespace utility
{

namespace audio
{

enum class FaderError
{
    OK = 0,         /**< Operation is success. */
    DONE,           /**< Processing of fading finalized. */
    UNSUPPORTED,    /**< The given data or file type is not supported. */
    FAILURE,        /**< Operation is failed. */
    NOTCONFIGURED   /**< Not Configured any formats, fade time etc. */
};

inline std::ostream& operator<<(std::ostream& os, FaderError error)
{
    const char* string[] = { "OK", "DONE", "UNSUPPORTED", "FAILURE", "NOTCONFIGURED" };
    return os << string[static_cast<int>(error)];
}

enum class FaderDir
{
    IN = -1,        /**< Fade IN */
    OFF = 0,        /**< Fade direction is OFF */
    OUT             /**< Fade OUT */
};

class Fader
{

public:

    Fader(LoggingInterface& loggingHandle);
    virtual ~Fader();

    /**
     * This function is called for configuring a fader for the specified parametrs.
     * This function can be called after setFadeDirectionAndTime() API to configure format, rate, channels
     * This function cannot be called during processing i.e when the state is in FADER_STATE_PROCESSING
     * @param[IN] format        Specify data format of stream is Little or Big endian, Signed or Unsigned 16-bit or some other type.
     * @param[IN] rate          The sample rate of the stream
     * @param[IN] channels      The channel count of the stream
     * @return OK               if fader is configured successfully
     * @return UNSUPPORTED      if specified parameters are of unspoorted formats and zero channels, Ramp calculation fails
     * @return FAILURE          if fade time or fade direction is not available and when state is not in FADER_STATE_PROCESSING, Ramp calculation fails
     */
    FaderError configure(const AudioFormat format, const uint32_t rate, const uint32_t channels);

    FaderError doConfigure();

    /**
     * This function will be called for configuring the FadeIn or FadeOut Time and direction
     * This is the first function to be called when fader object is created.
     * This function can be called during processing if user wants to configure different
     * fadetime and direction followed by restart() as mandatory
     * @param[IN] direction     Specify fader direction [FadeIn/FadeOut]
     * @param [IN] fadeTime     To configure the FadeIn/FadeOut time
     * @return NOTCONFIGURED    If fade direction is not IN/OUT
     * @return FAILURE          If fader is in FADER_STATE_PROCESSING state
     * @return E_OK             If fade direction and time was changed successfully
     */
    FaderError setFadeDirectionAndTime(const FaderDir direction, const uint32_t fadeTime);

    /**
     * This function is called for data processing [FadeIn/FadeOut]
     * This function is to be called after configure() API to start fade process
     * @param[IN] buf            Contains input PCM data for processing
     * @param[IN] frames         Count of frames for which fading will be applied [data in buf argument should match with frame count]
     * @param[OUT] outbuff       Contains Faded PCM data as output
     * @return FAILURE           If invalid input/output buffer, processing fails
     * @return NOTCONFIGURED     if fader is not in FADER_STATE_PROCESSING state
     * @return OK                If fade process is successful
     * @return DONE              If fade process is completed for given fade timeand state changes to FADER_STATE_COMPLETED
     */
    FaderError process(const void *buf, uint32_t frames, void *outbuff);

    /**
     * This function is called for restarting the Fader at anytime
     * This function can be called during process() i.e when the state is FADER_STATE_PROCESSING
     * to restart the fade process with same configured inputs for fadetime and direction.
     * To restart the fade process with different configured inputs for fadetime and direction,
     * restart() must be called after setFadeDirectionAndTime().
     * @return OK        if fader is restarted successfully, state moves to FADER_STATE_RESTARTED
     */
    FaderError restart();

    /**
     * This function returns the remaining frames for fading
     * @return mRemainingFrames:    mRemainingFrames for fading to complete
     */
    uint32_t getRemainingFrames();

    /**
     * This function will be called to get the Fade in or out Time
     * @param [OUT]fadeTime.  ..To get the configured FadeIn/FadeOut time
     * @return OK               If configured fade time is returned successfullyI
     */
    FaderError getFadeTime(uint32_t& fadeTime);

    /**
     * This function is called to reset fader at any time
     */
    FaderError reset();

private:
    typedef enum __fadestate
    {
        FADER_STATE_CONFIGURED  = 0x01,
        FADER_STATE_PROCESSING  = 0x02,
        FADER_STATE_STOPPED     = 0x04,
    } eState;

    uint32_t mRate;                /* Sampling rate */
    uint32_t mChannels;            /* Audio channels */
    uint32_t mFadeTime;            /* Fade time */
    uint32_t mRemainingFrames;     /* Remaining frames for fading */
    uint32_t mlevelOffset;         /* Level shift for unsigned formats */

    uint32_t mFadeFrames;          /* no. of fade frames for a configured fade-time*/
    uint32_t mFadeoutFrames;       /* Fade out frame count  */
    uint32_t *mCurve;              /* To hold Ramp values */

    FaderDir mDir;                 /* Fade Direction [FADE-IN, FADE_OUT]*/
    eState   mState;               /* States for fader */
    AudioFormat mFormat;           /* Audio Formats for fader */

    bool mSwap;
    bool mIsLe;

    LoggingInterface& mLoggingHandle;

    /**
     * This function is called for configuring default values to fader
     */
    void defaults();

    /**
     * This function is called for calculating the Linear Ramp values for applying fading
     * @param [IN/OUT] Fade Frames and pointer for holding calculated Ramp values
     * @return OK      if ramp calculation is success
     * @return FAILURE if ramp calculation is failed
     */
    FaderError calcLinearRamp();

    template<typename Func>
    FaderError calcRamp(Func func);

    /**
     * This function is called for checking the given format is supported by Fader or not
     * @param[IN] format    Specify data format of stream is Little or Big endian, Signed or Unsigned 8/16-bit or some other type.
     * @return OK           if the specified format is supported by Fader
     * @return UNSUPPORTED  if the specified parameters are not supported by Fader
     */
    FaderError isFormatSupported(const AudioFormat format);

    /**
     * This function is called by process function for applying the Ramp values on PCM data
     * @param[IN] buf         Contains input PCM data for processing
     * @param[IN] output      Contains Faded output PCM data
     * @param[IN] rampAddr    Contains the calculated Ramp values
     * @param[IN] frames      Count of frames for which fading will be applied [data in buf argument should match with frame count]
     * @return OK             if process of applying ramp on samples is success
     * @return DONE           if process of applying ramp on samples is completed for given fade time
     */
    template<typename T>
    FaderError processRampData(const T* buf, uint32_t* rampAddr, const uint32_t frames, T *output);

    template<typename E>
    void SwapBytes(E *value, unsigned int  size);

};

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOFADER_H_ */
